/* --COPYRIGHT--,BSD_EX
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *******************************************************************************
 *
 *                       MSP430 CODE EXAMPLE DISCLAIMER
 *
 * MSP430 code examples are self-contained low-level programs that typically
 * demonstrate a single peripheral function or device feature in a highly
 * concise manner. For this the code may rely on the device's power-on default
 * register values and settings such as the clock configuration and care must
 * be taken when combining code from several examples to avoid potential side
 * effects. Also see www.ti.com/grace for a GUI- and www.ti.com/msp430ware
 * for an API functional library-approach to peripheral configuration.
 *
 * --/COPYRIGHT--*/
//******************************************************************************
//  MSP430FR2311 Demo - UART-to-I2C Bridge
//
//  Description: This example enables serial communication between a UART device
//  and a I2C device. The hardware I2C interface utilizes the eUSCI_B module, and
//  the hardware UART utilizes the eUSCI_A module. Connect the I2C device to the
//  I2C interface (P1.2 and P1.3), and connect the UART device to the UART interface
//  (P1.6 and P1.7). This example supports full-duplex UART communication, eight
//  data bits with least significant bit first, no parity bit, and one stop bit.
//  After initializing everything, the CPU waits in LPM0 to save power until a I2C
//  or UART receive interrupt occurs. Then, the CPU exits LPM0, reads the I2C or
//  UART data, transmits the data on the other interface, and returns to LMP0.
//  ACLK = default REFO ~32768Hz, MCLK = SMCLK = 16MHz
//
//  How to use ?
//  Default UART Baud : 115200
//  Default I2C Clock : 100KHz
//  Read CMD  : 53 (Slave_addr << 1 + 1) Read_Length 50
//  Write CMD : 53 (Slave_addr << 1 + 0) Write_Length Data0  Data1 Data2 .... DataN 50
//  Repeated Write CMD : 53 (Slave_addr << 1 + 0) Write_Length Data0  Data1 Data2 .... DataN 53 (Slave_addr << 1 + 0) Write_Length Data0  Data1 Data2 .... DataN
//  Repeated Read CMD : 53 (Slave_addr << 1 + 0) Write_Length Data0  Data1 Data2 .... DataN 53 (Slave_addr << 1 + 1) Read_Length 50
//  Write Internal Register CMD : 57 Register0 Data0 Register1 Data1 ... 50
//  Read Internal Register CMD : 52 Register0 Register1 ... 50
//  Internal Register:
//                     0x00 : UART Baud Low Byte      // UART Baud = 16000000/Register_Value
//                     0x01 : UART Baud High Byte
//                     0x02 : I2C Clock Low Byte      // I2C Clock = 16000000/Register_Value
//                     0x03 : I2C Clock High Byte
//  Example : Read  : 53 91 08 50  ->  From Slave_addr 0x48 read 8 Byte
//            Write : 53 90 08 11 22 33 44 55 66 77 88 50  ->  Write Slave_addr 0x48 to 8 Byte (11 22 33 44 55 66 77 88)
//            Repeated Write : 53 90 08 11 22 33 44 55 66 77 88 53 90 03 99 AA BB 50 -> Write Slave_addr 0x48 to 3 Byte (99 AA BB) after Write Slave_addr 0x48 to 8 Byte (11 22 33 44 55 66 77 88)
//            Repeated Read : 53 90 08 11 22 33 44 55 66 77 88 53 91 08 50 -> From Slave_addr 0x48 read 8 Byte after Write Slave_addr 0x48 to 8 Byte (11 22 33 44 55 66 77 88)
//            Write Internal Register : 57 00 82 01 06 02 28 03 00 50 -> Write 0x8A to Register0(0x00)
//                                                                    -> Write 0x00 to Register1(0x01)
//                                                                    -> Write 0xA0 to Register2(0x02)
//                                                                    -> Write 0x00 to Register3(0x03)
//                                                                    -> Set UART Baud 115200, I2C Clock 100KHz
//            Read Internal Register : 52 00 01 02 03 50 -> Read Register0(0x00), Register1(0x01), Register2(0x02) and Register3(0x03) Value
//
//                                  MSP430FR2311
//                           ---------------------------
//                       /|\|                           |
//                        | |                           |
//                        --|RST                        |
//                          |                           |
//                          |                           |
//                          |                           |
//              I2C SDA  <->|P1.2/UCB0SDA   P1.7/UCA0TXD|--> UART TX Out
//                          |                           |
//              I2C SCL  <--|P1.3/UCB0SCL   P1.6/UCA0RXD|<-- UART RX In
//                          |                           |
//                          |                           |
//
//
//  Johnson He
//  Texas Instruments Inc.
//  June. 2019
//  Built with IAR Embedded Workbench v7.12.1 & Code Composer Studio v9.0.1
//******************************************************************************
#include <msp430.h>

// Define parameters
#define FRAM_ADDR 0xFD00                 // Internal FRAM address for store register
#define I2C_TO_UART_BUFF_SIZE 100        // BUFF_SIZE Can be modified if needed
#define UART_TO_I2C_BUFF_SIZE 100        // BUFF_SIZE Can be modified if needed

// Declare global variables
#if defined (__TI_COMPILER_VERSION__)
         unsigned int  i;                                         // for cycle
         char i2c_to_uart_data_buff[I2C_TO_UART_BUFF_SIZE];       // Buff
         char uart_to_i2c_data_buff[UART_TO_I2C_BUFF_SIZE];       // Buff
         unsigned char data_cnt;                                  // Used to data counter
volatile unsigned char uart_cmd ;                                 // Identify "S" "P"  S:0x53 P:0x50
volatile unsigned char i2c_send_cmd;                              // 1:Read 0:Write
volatile unsigned int  i2c_send_length1;                          // I2C send length
volatile unsigned int  i2c_send_length2;                          // I2C send length
volatile unsigned char i2c_send_cnt;                              // Used to I2C send data counter
volatile unsigned char i2c_recv_cnt;                              // Used to I2C recv data counter
volatile unsigned char uart_send_cnt;                             // Used to UART send data counter
volatile unsigned int  uart_baud;                                 // UART baud
volatile unsigned int  i2c_speed;                                 // I2C clock

#elif defined (__IAR_SYSTEMS_ICC__)
__no_init          unsigned int  i;                                    // for cycle
__no_init          char i2c_to_uart_data_buff[I2C_TO_UART_BUFF_SIZE];  // Buff
__no_init          char uart_to_i2c_data_buff[UART_TO_I2C_BUFF_SIZE];  // Buff
__no_init          unsigned char data_cnt;                             // Used to data counter
__no_init volatile unsigned char uart_cmd ;                            // Identify "S" "P"  S:0x53 P:0x50
__no_init volatile unsigned char i2c_send_cmd;                         // 1:Read 0:Write
__no_init volatile unsigned int  i2c_send_length1;                     // I2C send length
__no_init volatile unsigned int  i2c_send_length2;                     // I2C send length
__no_init volatile unsigned char i2c_send_cnt;                         // Used to I2C send data counter
__no_init volatile unsigned char i2c_recv_cnt;                         // Used to I2C recv data counter
__no_init volatile unsigned char uart_send_cnt;                        // Used to UART send data counter
__no_init volatile unsigned int  uart_baud;                            // UART baud
__no_init volatile unsigned int  i2c_speed;                            // I2C clock
#endif

void Bridge(void);

// Main Functions
int main(void)
{
    WDTCTL = WDTPW | WDTHOLD;                              // stop watchdog timer
    FRCTL0 = FRCTLPW | NWAITS_1;                           // Configure one FRAM waitstate as
                                                           // required by the device datasheet for
                                                           // MCLK operation beyond 8MHz _before_
                                                           // configuring the clock system
    __bis_SR_register(SCG0);                               // Disable FLL
    CSCTL3 |= SELREF__REFOCLK;                             // Set REFO as FLL reference source
    CSCTL0 = 0;                                            // clear DCO and MOD registers
    CSCTL1 &= ~(DCORSEL_7);                                // Clear DCO frequency select bits first
    CSCTL1 |= DCORSEL_5;                                   // Set DCO = 16MHz
    CSCTL2 = FLLD_0 + 487;                                 // DCOCLKDIV = 16MHz
    __delay_cycles(3);                                     // Wait
    __bic_SR_register(SCG0);                               // Enable FLL
    while(CSCTL7 & (FLLUNLOCK0 | FLLUNLOCK1));             // FLL locked

    CSCTL4 = SELMS__DCOCLKDIV | SELA__REFOCLK;             // Set default REFO (~32768Hz)
                                                           // as ACLK source, ACLK = 32768Hz
                                                           // Default DCOCLKDIV as MCLK
                                                           // and SMCLK source

    PM5CTL0 &= ~LOCKLPM5;                                  // Disable GPIO power-on default
                                                           // high-impedance mode

    // Read data from FRAM used to UART baud and I2C clock
    uart_baud = *(unsigned int*) FRAM_ADDR;                // UART baud address, 16 bits
    i2c_speed = *(unsigned int*) (FRAM_ADDR + 2);          // I2C clock address, 16 bits

    // Initialize hardware I2C pins
    P1SEL0 |= BIT2 | BIT3 | BIT6 | BIT7;                   // I2C and UART pins

    // Configure eUSCI_1 UART module
    UCA0CTLW0 |= UCSWRST;                                  // Software reset enabled
    UCA0CTLW0 |= UCSSEL_2;                                 // set SMCLK as BRCLK
    UCA0BRW = (uart_baud == 0xffff) ? 0x008A : uart_baud;  // baud = INT(16000000/uart_baud), default : 115200
    UCA0CTLW0 &= ~UCSWRST;                                 // Initialize eUSCI
    UCA0IE |= UCRXIE;                                      // Enable USCI_A0 RX interrupt

    // Configure eUSCI_B I2C module
    UCB0CTLW0 |= UCSWRST;                                  // Software reset enabled
    UCB0CTLW0 |= UCMODE_3 | UCMST | UCSYNC;                // I2C mode, Master mode, sync
    UCB0CTLW1 |= UCASTP_1;                                 // No automatic stop generated
    UCB0BRW = (i2c_speed == 0xffff) ? 0x00A0 : i2c_speed;  // Bit rate clock = SMCLK/i2c_speed,default : 100KHz
    UCB0CTLW0 &= ~UCSWRST;                                 // Initialize eUSCI
    UCB0IE |= UCRXIE | UCTXIE0 | UCBCNTIE | UCSTPIE | UCNACKIE;       // Enable USCI_B0 RX TX CNT interrupt

    // Enable global interrupts and enter LPM0
    __bis_SR_register(LPM0_bits + GIE);
}

#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector=USCI_A0_VECTOR
__interrupt void USCI_A0_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(USCI_A0_VECTOR))) USCI_A0_ISR (void)
#else
#error Compiler not supported!
#endif
{
      if(uart_cmd == 0)
      {
         uart_to_i2c_data_buff[0] = UCA0RXBUF;
         data_cnt = 0;
         switch (UCA0RXBUF)
         {
         case 0x53: uart_cmd = 1; break;     // I2C Write/Read/Repeated_Write/Repeated_Read mode
         case 0x52: uart_cmd = 2; break;     // Read internal register mode
         case 0x57: uart_cmd = 3; break;     // Write internal register mode
         default:   uart_cmd = 0; break;     // don't care
         }
      }
      else
      {
          Bridge();                          // bridge function
      }
}

#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector = USCI_B0_VECTOR
__interrupt void USCIB0_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(USCI_B0_VECTOR))) USCIB0_ISR (void)
#else
#error Compiler not supported!
#endif
{
  switch(__even_in_range(UCB0IV, USCI_I2C_UCBIT9IFG))
  {
    case USCI_I2C_UCRXIFG0:                       // Vector 22: RXIFG0
         i2c_to_uart_data_buff[i2c_recv_cnt] = UCB0RXBUF;
         i2c_recv_cnt ++;
         break;
    case USCI_I2C_UCNACKIFG:
         uart_cmd = 0;
         UCB0CTLW0 |= UCTXSTP;
         break;         // Vector 4: NACKIFG
    case USCI_I2C_UCTXIFG0:                       // Vector 24: TXIFG0
         UCB0TXBUF = uart_to_i2c_data_buff[i2c_send_cnt++]; // Send data by I2C
         break;
    case USCI_I2C_UCSTPIFG:                       // Vector 8: STPIFG
         UCB0IFG &= ~UCSTPIFG;                    // Clear stop condition int flag
         while(i2c_recv_cnt--)
         {
             while(!(UCA0IFG&UCTXIFG));           // Wait UCA0 free
             UCA0TXBUF = i2c_to_uart_data_buff[uart_send_cnt++];
         }
         uart_send_cnt = 0;
         i2c_recv_cnt = 0;
         break;
    case USCI_I2C_UCBCNTIFG:                      // Vector 26: BCNTIFG
         if(i2c_send_length2 == 0)
         {
             UCB0CTLW0 |= UCTXSTP;                // Send STOP Bit
             i2c_send_cmd = 0;
             i2c_send_cnt = 0;
         }
         else
         {
             if(i2c_send_cmd == 3)
             {
                 UCB0CTLW0 &= ~UCTR;                                                  // Receiver/Read
             }
             else if(i2c_send_cmd == 4)
             {
                 UCB0CTLW0 |= UCTR;                                                   // Transmitter/Write
                 i2c_send_cnt = i2c_send_length1 + 7;
             }
             UCB0I2CSA = (uart_to_i2c_data_buff[i2c_send_length1 + 4] & 0xfe) >> 1;   // Slave address
             UCB0TBCNT = i2c_send_length2;                                            // I2C write length
             UCB0TXBUF = uart_to_i2c_data_buff[i2c_send_length1 + 6];                 // I2C send data
             UCB0CTL1 |= UCTXSTT;                                                     // I2C start condition
             i2c_send_length2 = 0;
         }
         break;
  }
}

void Bridge(void)
{
    switch(uart_cmd)
    {
    case 1:    // I2C Write/Read/Repeated_Write/Repeated_Read
           data_cnt ++ ;
           uart_to_i2c_data_buff[data_cnt] = UCA0RXBUF;
           i2c_send_cmd = (i2c_send_cmd == 0) ? (((UCA0RXBUF & 0x01) == 1) ? 1 : 2) : i2c_send_cmd; // 1:Read 2:Write/Repeated Write/Repeated Read
           switch(i2c_send_cmd)
           {
           // Read cmd
           case 1:
           if(data_cnt == 3)
           {
               i2c_send_length1 = uart_to_i2c_data_buff[2];                   // Get read data length
               if(uart_to_i2c_data_buff[3] == 0x50)
               {
                   uart_cmd = 0;                                              // Exit cmd
                   i2c_send_length2 = 0;
                   data_cnt = 0;
                   UCB0CTLW0 &= ~UCTR;                                        // Receiver/Read
                   UCB0I2CSA = (uart_to_i2c_data_buff[1] & 0xfe) >> 1;        // Slave address
                   UCB0TBCNT = i2c_send_length1;                              // I2C read length
                   UCB0CTL1 |= UCTXSTT;                                       // I2C start condition
               }
           }
           break;

           // Write cmd
           case 2:
           if((data_cnt > 2) && (data_cnt == uart_to_i2c_data_buff[2] + 3))
           {
               i2c_send_length1 = uart_to_i2c_data_buff[2];                   // Get write data length
               if(uart_to_i2c_data_buff[i2c_send_length1 + 3] == 0x50)
               {
                   uart_cmd = 0;                                              // Exit cmd
                   i2c_send_length2 = 0;
                   data_cnt = 0;
                   UCB0CTLW0 |= UCTR;                                         // Transmitter/Write
                   UCB0I2CSA = (uart_to_i2c_data_buff[1] & 0xfe) >> 1;        // Slave address
                   UCB0TBCNT = i2c_send_length1;                              // I2C write length
                   UCB0CTL1 |= UCTXSTT;                                       // I2C start condition
                   i2c_send_cnt = 3;
               }
               else if(uart_to_i2c_data_buff[i2c_send_length1 + 3] == 0x53)
               {
                   i2c_send_cmd = 3;                                          // Enter Repeated Read mode
               }
           }
           break;

           // Repeated Read cmd
           case 3:
           if(data_cnt == i2c_send_length1 + 6)
           {
               if((uart_to_i2c_data_buff[i2c_send_length1 + 4] & 0x01) == 0)
               {
                   i2c_send_cmd = 4;                                          // Enter Repeated Write mode
                   i2c_send_length2 = uart_to_i2c_data_buff[i2c_send_length1 + 5];
               }
               else if(uart_to_i2c_data_buff[i2c_send_length1 + 6] == 0x50)
               {
                   uart_cmd = 0;                                              // Exit cmd
                   i2c_send_length2 = uart_to_i2c_data_buff[i2c_send_length1 + 5];
                   data_cnt = 0;
                   UCB0CTLW0 |= UCTR;                                         // Transmitter/Write
                   UCB0I2CSA = (uart_to_i2c_data_buff[1] & 0xfe) >> 1;        // Slave address
                   UCB0TBCNT = i2c_send_length1;                              // I2C write length
                   UCB0CTL1 |= UCTXSTT;                                       // I2C start condition
                   i2c_send_cnt = 3;
               }
           }
           break;

           // Repeated Write cmd
           case 4:
           if(data_cnt == i2c_send_length1 + i2c_send_length2 + 6)
           {
               if(uart_to_i2c_data_buff[i2c_send_length1 + i2c_send_length2 + 6] == 0x50)
               {
                    uart_cmd = 0;                                             // Exit cmd
                    data_cnt = 0;
                    UCB0CTLW0 |= UCTR;                                        // Transmitter/Write
                    UCB0I2CSA = (uart_to_i2c_data_buff[1] & 0xfe) >> 1;       // Slave address
                    UCB0TBCNT = i2c_send_length1;                             // I2C write length
                    UCB0CTL1 |= UCTXSTT;                                      // I2C start condition
                    i2c_send_cnt = 3;
               }
           }
           break;

           default: break;
           }
           break;
    case 2:    // Read internal register
          data_cnt ++ ;
          uart_to_i2c_data_buff[data_cnt] = UCA0RXBUF;
          if(uart_to_i2c_data_buff[data_cnt] == 0x50)      // STOP
          {
              for(i=1;i<data_cnt;i++)
                  {
                  while(!(UCA0IFG&UCTXIFG));
                  UCA0TXBUF = *(unsigned char *)(FRAM_ADDR + uart_to_i2c_data_buff[i]);     // Output value of internal register
                  }
              uart_cmd = 0;    // Exit cmd
          }
        break;
    case 3:    // Write internal register
           data_cnt ++ ;
           uart_to_i2c_data_buff[data_cnt] = UCA0RXBUF;
           if((data_cnt % 2 == 1) && (uart_to_i2c_data_buff[data_cnt] == 0x50))   // STOP
           {
               SYSCFG0 = FRWPPW;
               for(i=1;i<data_cnt;i+=2)
                   {
                   *(unsigned char *)(FRAM_ADDR + uart_to_i2c_data_buff[i]) = uart_to_i2c_data_buff[i+1];  // Write value of internal register
                   }
               SYSCFG0 = FRWPPW | PFWP;
               UCA0CTLW0 |= UCSWRST;
               UCA0BRW = (*(unsigned int*) FRAM_ADDR == 0xffff) ? 0x008A : *(unsigned int*) FRAM_ADDR;     // Set UART baud
               UCA0CTLW0 &= ~UCSWRST;
               UCA0IE |= UCRXIE;
               UCB0CTLW0 |= UCSWRST;
               UCB0BRW = (*(unsigned int*)(FRAM_ADDR + 2) == 0xffff) ? 0x00A0 : *(unsigned int*) (FRAM_ADDR + 2);  // Set I2C clock
               UCB0CTLW0 &= ~UCSWRST;
               UCB0IE |= UCRXIE | UCTXIE0 | UCBCNTIE | UCSTPIE;
               uart_cmd = 0;   // Exit cmd
           }
        break;
    default : break;
    }
}
