/*******************************************************************************
* Name: pwm.c 
* 
* Description: Functions to control the PWM outputs by configuring the 
* duty cycle. 
* 
* Texas Instruments, Inc
* 
* Version: 1.0
*******************************************************************************/

/*
 * Copyright (C) {2011} Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

// Include Library Headers
#include "device.h"
#include "pwm.h"
#include "misc.h"

/* ****************************************************************************
 * Function Name: PWM_Control
 * 
 * Description: Outputs a 20kHz PWM on the channel selected by pwm. Duty cycle 
 * adjustable from 0 to 1023. To specify 80% duty cycle on Battery 1, Voltage 
 * Channel: PWM_Control(BATT_1, CHANNEL_VOLTAGE, PWM_ON, 0.8 * PWM_DUTY_100);
 * where pwm_duty_period = 0.8 * PWM_DUTY_100
 * 
 * Inputs:
 * o batt_num: Specify which battery. Notation: BATT_1, BATT_2
 * o pwm_channel: Specify which PWM channel number: voltage/current  
 * Notation: CHANNEL_VOLTAGE, CHANNEL_CURRENT
 * o on_off: Notation: PWM_OFF, PWM_ON
 * o pwm_duty_period: Notation: PWM_DUTY_0, ..., PWM_DUTY_100
 * ***************************************************************************/
void PWM_Control(unsigned char batt_num, unsigned char pwm_channel, \
	unsigned char on_off, unsigned int pwm_duty_period)
{
	
	if(batt_num == BATT_1)
	{
		if(pwm_channel == CHANNEL_VOLTAGE)			// V_PWM_1 = P1.2 = TA0.1
		{
			if(on_off == PWM_ON)
			{
				// Set PWM output to Reset/Set Mode
				TA0CCTL1 = OUTMOD_7;
				TA0CCR1 = pwm_duty_period;
			}
			else
			{
				// Set PWM output to LOW state
				TA0CCTL1 &= ~(OUT);
				TA0CCTL1 = OUTMOD_0;
				TA0CCR1 = PWM_DUTY_0;
			}
		}
		else if(pwm_channel == CHANNEL_CURRENT)		// I_PWM_1 = P1.3 = TA0.2
		{
			if(on_off == PWM_ON)
			{
				// Set PWM output to Reset/Set Mode
				TA0CCTL2 = OUTMOD_7;				
				TA0CCR2 = pwm_duty_period;
			}
			else
			{
				// Set PWM output to LOW state
				TA0CCTL2 &= ~(OUT);	
				TA0CCTL2 = OUTMOD_0;
				TA0CCR2 = PWM_DUTY_0;
			}
		}
	}
	else if(batt_num == BATT_2)						
	{	
		if(pwm_channel == CHANNEL_VOLTAGE)			// V_PWM_2 = P1.4 = TA0.3
		{				
			if(on_off == PWM_ON)
			{
				// Set PWM output to Reset/Set Mode
				TA0CCTL3 = OUTMOD_7;				
				TA0CCR3 = pwm_duty_period;
			}
			else
			{
				// Set PWM output to LOW state
				TA0CCTL3 &= ~(OUT);	
				TA0CCTL3 = OUTMOD_0;		
				TA0CCR3 = PWM_DUTY_0;
			}
		}
		else if(pwm_channel == CHANNEL_CURRENT)		// I_PWM_2 = P1.5 = TA0.4
		{			
			if(on_off == PWM_ON)
			{
				// Set PWM output to Reset/Set Mode
				TA0CCTL4 = OUTMOD_7;				
				TA0CCR4 = pwm_duty_period;
			}
			else
			{
				// Set PWM output to LOW state
				TA0CCTL4 &= ~(OUT);
				TA0CCTL4 = OUTMOD_0;
				TA0CCR4 = PWM_DUTY_0;
			}
		}	
		
	}		
	
}
